unit Sprites;

interface

uses
  Windows, SysUtils, Classes, Graphics, Controls, TurboSprite,
  Grafix;

type

  TDIBSprite = class( TSprite )
  private
    FFrameX: integer;
    FFrameY: integer;
    bFrames: boolean;
  protected
    FDIB: TDIB;
    Width2, Height2: integer;
  public
    constructor CreateDIBSprite( MoveProc: TMoveProc; dib: TDIB );
    procedure Render; override;
    property DIB: TDIB read FDIB write FDIB;
    property FrameX: integer read FFrameX write FFrameX;
    property FrameY: integer read FFrameY write FFrameY;
  end;

  TAnimatedDIBSprite = class( TDIBSprite )
  private
    FAnim: boolean;
    FInterval: integer;
    Counter: integer;
    FKill: boolean;
    FCycleY: boolean;
  protected
  public
    constructor CreateAnimatedDIBSprite( MoveProc: TMoveProc; dib: TDIB );
    procedure Move; override;
  published
    property Animated: boolean read FAnim write FAnim default true;
    property AnimationInterval: integer read FInterval write FInterval;
    property CycleY: boolean read FCycleY write FCycleY;
    property KillWhenComplete: boolean read FKill write FKill;
  end;

  TPolygonSprite = class( TSprite )
  private
     r: TRect;
     OldAngle: integer;
     FFilled: boolean;
  protected
     poly: TPolygon;
     polyOriginal: TPolygon;
     FColorIndex: byte;
     procedure setSize; virtual;
     procedure addVertices; virtual; abstract;
  public
     constructor CreatePolygonSprite( MoveProc: TMoveProc; const nCol: byte );
     destructor Destroy; override;
     procedure Render; override;
     property ColorIndex: byte read FColorIndex write FColorIndex;
     property Filled: boolean read FFilled write FFilled;
  end;

  TSquareSprite = class( TPolygonSprite )
  private
  protected
     procedure AddVertices; override;
  public
  end;

implementation

(***************************************************
TDIBSprite
***************************************************)
constructor TDIBSprite.CreateDIBSprite( MoveProc: TMoveProc; dib: TDIB );
begin
  inherited Create( MoveProc );
  FDIB := dib;
  bFrames := ( dib.FramesX > 0 ) and ( dib.FramesY > 0 );
  if bFrames then
  begin
    Width := dib.DIBCanvas.Width div dib.FramesX;
    Height := dib.DIBCanvas.Height div dib.FramesY;
  end
  else
  begin
    Width := dib.DIBCanvas.Width;
    Height := dib.DIBCanvas.Height;
  end;
  Width2 := Width div 2;
  Height2 := Height div 2;
end;

procedure TDIBSprite.Render;
var
  rectDest, rectSource: TRect;
begin
  inherited Render;
  if Visible then
  begin
    with rectDest do
    begin
      Left := ptPhysical.X - Width2;
      Top := ptPhysical.Y - Height2;
      Right := Left + Width;
      Bottom := Top + Height;
    end;
    with rectSource do
    begin
      Left := FrameX * Width;
      Top := FrameY * Height;
      Right := Left + Width;
      Bottom := Top + Height;
    end;
    SpriteSurface.DIBCanvas.CopyRect( rectDest, FDIB.DIBCanvas, rectSource, FDIB.Transparent );
  end;
end;

(*********************************************
TAnimatedDIBSprite
*********************************************)
constructor TAnimatedDIBSprite.CreateAnimatedDIBSprite( MoveProc: TMoveProc; dib: TDIB );
begin
  inherited CreateDIBSprite( MoveProc, dib );
  FAnim := true;
end;

procedure TAnimatedDIBSprite.Move;
begin
  inherited Move;
  if FAnim then
  begin
    Inc( Counter );
    if Counter >= FInterval then
    begin
      Counter := 0;
      FrameX := FrameX + 1;
      if FrameX >= FDIB.FramesX then
      begin
        FrameX := 0;
        if (FDIB.FramesY <= 1) or not FCycleY then
          if FKill then
          begin
            Visible := false;
            Dead := true;
            Exit;
          end;
        FrameX := 0;
        if FCycleY then
        begin
          FrameY := FrameY + 1;
          if FrameY >= FDIB.FramesY then
          begin
            FrameY := 0;
            if FKill then
            begin
              Visible := false;
              Dead := true;
            end;
          end;
        end;
      end;
    end;
  end;
end;

(*********************************************
TPolygonSprite
*********************************************)
constructor TPolygonSprite.CreatePolygonSprite( MoveProc: TMoveProc; const nCol: byte );
begin
  inherited Create( MoveProc );
  FColorIndex := nCol;
  poly := TPolygon.Create;
  AddVertices;
  r := Rect( 0, 0, 0, 0 );
  SetSize;
  polyOriginal := TPolygon.Create;
  polyOriginal.Assign( poly );
  Angle := 90;
end;

destructor TPolygonSprite.Destroy;
var
  i: integer;
begin
  polyOriginal.Free;
  poly.Free;
  inherited Destroy;
end;

procedure TPolygonSprite.Render;
var
  i: integer;
  ptFrom, ptTo: TPoint;
begin
  inherited Render;
  if Angle <> OldAngle then
  begin
    poly.Assign( polyOriginal );
    poly.Rotate( Angle );
    SetSize;
  end;
  OldAngle := Angle;
  with SpriteSurface.DIBCanvas do
  begin
    PenColorIndex := FColorIndex;
    BrushColorIndex := FColorIndex;
  end;
  if FFilled then
    SpriteSurface.DIBCanvas.fillPolygon( ptPhysical.X, ptPhysical.Y, poly )
  else
    SpriteSurface.DIBCanvas.polygon( ptPhysical.X, ptPhysical.Y, poly );
end;

procedure TPolygonSprite.SetSize;
var
  i: integer;
begin
  r.Left := 0;
  r.Right := 0;
  r.Top := 0;
  r.Bottom := 0;
  for i := 0 to poly.NumVertex - 1 do
     with poly.pts[i] do
        begin
           if X < r.Left then
              r.Left := X;
           if Y < r.Top then
              r.Top := Y;
           if X > r.Right then
              r.Right := X;
           if Y > r.Bottom then
              r.Bottom := Y;
        end;
  Width := r.Right - r.Left + 10;
  Height := r.Bottom - r.Top + 10;
end;

(*********************************************
TSquareSprite
*********************************************)
procedure TSquareSprite.addVertices;
begin
  with poly do
     begin
        addPoint( Point( -5, -5 ) );
        addPoint( Point( 5, -5 ) );
        addPoint( Point( 5, 5 ) );
        addPoint( Point( -5, 5 ) );
     end;
end;

end.
